//*************************************************************************************************
//
//	Description:
//		Skyring
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Alastair Murray
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray			23/10/2007  0.1				Created (copied from Unlit.fx)
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif

//
// Transforms
//
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap											// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE											// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useDepthMap, "Use depthmap?", "USE_DEPTH_MAP" )	// TRUE if a depth map should be used to modulate the fog

#if defined( _3DSMAX_ ) || defined( USE_DEPTH_MAP )
DEPENDENT_TEXTURE_PARAM( depthTexture, "Depth Tex {UV1}", useDepthMap )
#endif

float4 staticColour : Color												// static colour multiplier
<
	string UIName = "Static Colour";
	string UIType = "ColorSwatch";
	bool appEdit = true;
> = {1.0f, 1.0f, 1.0f, 1.0f};

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


#if defined( _3DSMAX_ ) || defined( USE_DEPTH_MAP )
sampler2D depthMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="depthTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < depthTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Functions
//




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position	: POSITION;														// View-coords position
	float4 colour		: TEXCOORD1;															// Vertex colour
	float2 texCoord	: TEXCOORD0;													// UV coords for texture channel 0
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT SkyringVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	// Copy simple invariant params to output structure
	_output.colour	 = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex (forced to the farplane)
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj ).xyww;

	// If we don't subtract a small epsilon from the z, some graphics cards' inaccurate
	// calcs result in pixels bing clipped past the farZ
	_output.position.z -= 0.00001f;

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector from the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = worldPos - viewI[ 3 ];
#else
	float3 worldEyeVec = worldPos - worldCameraPos;
#endif

	float3 normal = normalize( float3( -worldEyeVec.x, 1.0f, -worldEyeVec.z ) );

	// Perform lighting calcs
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
struct PSINPUT
{
	float4 colour		: TEXCOORD1;													// Vertex colour
	float2 texCoord		: TEXCOORD0;										// UV coords for texture channel 0
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkyringFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	// Calculate pixel colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;

	// Calculate fog factor
	float fogFactor;

	DEPENDENT_CODE_START( useDepthMap )
#if defined( _3DSMAX_ ) || defined( USE_DEPTH_MAP )
	// Depth map: per pixel fog setting to give flat skyrings a feeling of variable distance
	fogFactor = tex2D( depthMap, _input.texCoord ).g;
#endif
	DEPENDENT_CODE_ELSE( useDepthMap )
#if defined( _3DSMAX_ ) || !defined( USE_DEPTH_MAP )
	// No depth map: use the default fog amount
	fogFactor = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Distance fog is applied according to the fog factor, from 0.0 = "standard" max fog amount, to 1.0 = full fog
	APPLY_FAR_DISTANCE_FOG( accumulatedColour, fogFactor );
	_output.Colour = CalculateOutputPixel(accumulatedColour);
	

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkyringLowDetailFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	// Calculate pixel colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;

	// Calculate fog factor
	float fogFactor;

	DEPENDENT_CODE_START( useDepthMap )
#if defined( _3DSMAX_ ) || defined( USE_DEPTH_MAP )
	// Depth map: per pixel fog setting to give flat skyrings a feeling of variable distance
	fogFactor = tex2D( depthMap, _input.texCoord ).g;
#endif
	DEPENDENT_CODE_ELSE( useDepthMap )
#if defined( _3DSMAX_ ) || !defined( USE_DEPTH_MAP )
	// No depth map: use the default fog amount
	fogFactor = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )

	// Distance fog is applied according to the fog factor, from 0.0 = "standard" max fog amount, to 1.0 = full fog
	APPLY_FAR_DISTANCE_FOG( accumulatedColour, fogFactor );
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);
	

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkyringZPrimeDOFFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	_output.Colour.rgb = 1.0f;	// max distance

	_output.Colour.a = diffuseTexColour.w;

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Skyring
<
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Skyring";
	int    normalDeferredID		= 1;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER";
	string zprimeDOFTechnique	= "_Skyring_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Skyring_LowDetail";
	int    lowDetailDeferredID = 1;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkyringVertexShader();
		PixelShader = compile sce_fp_rsx SkyringFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkyringVertexShader();
		PixelShader = compile ps_3_0 SkyringFragmentShader();
#endif		
	}
}



technique Skyring_Trans
<
    bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Skyring";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER";
	string zprimeDOFTechnique	= "_Skyring_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Skyring_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 SkyringVertexShader();
		PixelShader = compile ps_3_0 SkyringFragmentShader();
#endif		
	}
}



technique Skyring_Trans_NoDefer
<
    bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Skyring";
	int    normalDeferredID		= 1;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER";
	string zprimeDOFTechnique	= "_Skyring_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Skyring_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 SkyringVertexShader();
		PixelShader = compile ps_3_0 SkyringFragmentShader();
#endif		
	}
}


technique _Skyring_ZPrime_DOF
{
	pass Pass0
	{
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;

#if defined (_PS3_)
		AlphaFunc = int2(GEqual, 0x40);
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx SkyringVertexShader();
		PixelShader = compile sce_fp_rsx SkyringZPrimeDOFFragmentShader();
#else		
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
		CullMode = None;
		VertexShader = compile vs_3_0 SkyringVertexShader();
		PixelShader = compile ps_3_0 SkyringZPrimeDOFFragmentShader();
#endif
	}
}


// Inherits renderstate from the high detail technique
technique _Skyring_LowDetail
<
  bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkyringVertexShader();
		PixelShader = compile sce_fp_rsx SkyringLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkyringVertexShader();
		PixelShader = compile ps_3_0 SkyringLowDetailFragmentShader();
#endif		
	}
}
